'use client'

import { useEffect, useMemo } from 'react'
import { useSession } from 'next-auth/react'
import type { Session } from 'next-auth'

type StoredAuth = {
  user: Record<string, any>
  sessionToken: string
  refreshToken: string
  business: boolean
  remember?: boolean
}

const buildStoredAuth = (session: Session | null | undefined): StoredAuth | null => {
  const user = session?.user

  if (!user?.sessionToken || !user.refreshToken) {
    return null
  }

  return {
    user: {
      userId: user.id,
      _id: user.id,
      name: user.name || '',
      firstName: (user as any).firstName || '',
      middleName: (user as any).middleName || '',
      lastName: (user as any).lastName || '',
      role: (user as any).role || 'DEFAULT',
      businessRole: (user as any).businessRole || 'OWNER',
      email: user.email || '',
      phoneNumber: (user as any).phoneNumber || '',
      dob: (user as any).dob || '',
      username: user.email || '',
      imageUrl: (user as any).imageUrl || '',
      gender: (user as any).gender || 'NOT_SPECIFY',
      country: (user as any).country || 'IND'
    },
    sessionToken: user.sessionToken,
    refreshToken: user.refreshToken,
    business: (user as any).business ?? false,
    remember: true
  }
}

const saveAuthPayload = (payload: StoredAuth) => {
  const serialized = JSON.stringify(payload)
  localStorage.setItem('currentAuth', serialized)
  sessionStorage.setItem('currentAuth', serialized)
}

const clearAuthPayload = () => {
  localStorage.removeItem('currentAuth')
  sessionStorage.removeItem('currentAuth')
}

const AuthStorageSync = () => {
  const sessionResult = useSession()
  const { data: session, status } = sessionResult

  const payload = useMemo(() => buildStoredAuth(session), [session])

  useEffect(() => {
    if (typeof window === 'undefined') return

    if (payload) {
      saveAuthPayload(payload)
    } else if (status === 'unauthenticated') {
      clearAuthPayload()
    }
  }, [payload, status])

  return null
}

export default AuthStorageSync

