'use client';

import { useEffect, useState } from 'react';

import axios from 'axios';
import { Card, CardContent, Container, Divider, List, ListItem, ListItemText, Typography } from '@mui/material';

interface BoxingClub {
  id: string;
  title: string;
  createdAt: string;
}

const API_BASE_URL = process.env.NEXT_PUBLIC_API_URI || 'http://localhost:3024';

const BoxingClubDirectory = () => {
  const [clubs, setClubs] = useState<BoxingClub[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchClubs = async () => {
      try {
        setLoading(true);
        const response = await axios.get(`${API_BASE_URL}/boxing-clubs/public`);
        if (response.data.success) {
          setClubs(response.data.data);
        }
      } catch (error) {
        console.error('Failed to load boxing clubs', error);
      } finally {
        setLoading(false);
      }
    };

    fetchClubs();
  }, []);

  return (
    <Container maxWidth='md' sx={{ py: 8 }}>
      <Typography variant='h3' component='h1' gutterBottom>
        🥊 Boxing Clubs
      </Typography>
      <Typography variant='subtitle1' color='text.secondary' sx={{ mb: 4 }}>
        Discover the boxing clubs curated by our admins.
      </Typography>

      {loading ? (
        <Typography>Loading boxing clubs...</Typography>
      ) : clubs.length === 0 ? (
        <Typography>No boxing clubs are available yet. Please check back later.</Typography>
      ) : (
        <Card>
          <CardContent>
            <List>
              {clubs.map((club, index) => (
                <div key={club.id}>
                  <ListItem>
                    <ListItemText
                      primary={
                        <Typography variant='h6' component='span'>
                          {club.title}
                        </Typography>
                      }
                      secondary={
                        club.createdAt ? `Added on ${new Date(club.createdAt).toLocaleDateString()}` : undefined
                      }
                    />
                  </ListItem>
                  {index < clubs.length - 1 && <Divider component='li' />}
                </div>
              ))}
            </List>
          </CardContent>
        </Card>
      )}
    </Container>
  );
};

export default BoxingClubDirectory;

