'use client';

import { useEffect, useState } from 'react';

import { useRouter } from 'next/navigation';

import {
  Card,
  CardContent,
  Container,
  Divider,
  List,
  ListItem,
  ListItemText,
  Typography
} from '@mui/material';

import useAPI from '@/hooks/useAPI';

interface Professional {
  id: string;
  title: string;
  createdAt: string;
}

const ProfessionalDirectory = () => {
  const router = useRouter();
  const [professionals, setProfessionals] = useState<Professional[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    const fetchProfessionals = async () => {
      try {
        setLoading(true);
        // eslint-disable-next-line react-hooks/rules-of-hooks
        const api = await useAPI();
        const response = await api.professional.getPublic();
        if (response.data.success) {
          setProfessionals(response.data.data);
        }
      } catch (err: any) {
        console.error('Error fetching professionals', err);
        if (err.response?.status === 401) {
          setError('Please login to view the professionals list.');
          setTimeout(() => router.push('/login'), 1500);
        } else {
          setError(err.response?.data?.message || 'Failed to load professionals.');
        }
      } finally {
        setLoading(false);
      }
    };

    fetchProfessionals();
  }, [router]);

  return (
    <Container maxWidth='md' sx={{ py: 8 }}>
      <Typography variant='h3' component='h1' gutterBottom>
        👩‍⚖️ Registered Professionals
      </Typography>
      <Typography variant='subtitle1' color='text.secondary' sx={{ mb: 4 }}>
        Browse the list of professionals available to authenticated members.
      </Typography>

      {loading ? (
        <Typography>Loading professionals...</Typography>
      ) : error ? (
        <Typography color='error'>{error}</Typography>
      ) : professionals.length === 0 ? (
        <Typography>No professionals are available yet. Please check back later.</Typography>
      ) : (
        <Card>
          <CardContent>
            <List>
              {professionals.map((professional, index) => (
                <div key={professional.id}>
                  <ListItem>
                    <ListItemText
                      primary={
                        <Typography variant='h6' component='span'>
                          {professional.title}
                        </Typography>
                      }
                      secondary={
                        professional.createdAt
                          ? `Added on ${new Date(professional.createdAt).toLocaleDateString()}`
                          : undefined
                      }
                    />
                  </ListItem>
                  {index < professionals.length - 1 && <Divider component='li' />}
                </div>
              ))}
            </List>
          </CardContent>
        </Card>
      )}
    </Container>
  );
};

export default ProfessionalDirectory;

