'use client';

import { useEffect, useState } from 'react';

import axios from 'axios';
import {
  Card,
  CardContent,
  Container,
  Divider,
  List,
  ListItem,
  ListItemText,
  Typography,
} from '@mui/material';

interface Certificate {
  id: string;
  title: string;
  createdAt: string;
}

const API_BASE_URL = process.env.NEXT_PUBLIC_API_URI || 'http://localhost:3024';

const PublicCertificatesList = () => {
  const [certificates, setCertificates] = useState<Certificate[]>([]);
  const [loading, setLoading] = useState<boolean>(true);

  useEffect(() => {
    const fetchCertificates = async () => {
      try {
        const response = await axios.get(`${API_BASE_URL}/certificates/public`);

        if (response.data.success) {
          setCertificates(response.data.data);
        }
      } catch (error) {
        console.error('Error fetching certificates:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchCertificates();
  }, []);

  return (
    <Container maxWidth='md' sx={{ py: 8 }}>
      <Typography variant='h3' component='h1' gutterBottom>
        🎖️ Official Certificates
      </Typography>
      <Typography variant='subtitle1' color='text.secondary' sx={{ mb: 4 }}>
        Browse the certificates published by the administrators.
      </Typography>

      {loading ? (
        <Typography>Loading certificates...</Typography>
      ) : certificates.length === 0 ? (
        <Typography>No certificates are available yet. Please check back later.</Typography>
      ) : (
        <Card>
          <CardContent>
            <List>
              {certificates.map((certificate, index) => (
                <div key={certificate.id}>
                  <ListItem>
                    <ListItemText
                      primary={
                        <Typography variant='h6' component='span'>
                          {certificate.title}
                        </Typography>
                      }
                      secondary={
                        certificate.createdAt
                          ? `Published on ${new Date(certificate.createdAt).toLocaleDateString()}`
                          : undefined
                      }
                    />
                  </ListItem>
                  {index < certificates.length - 1 && <Divider component='li' />}
                </div>
              ))}
            </List>
          </CardContent>
        </Card>
      )}
    </Container>
  );
};

export default PublicCertificatesList;

