'use client';

import { useState, useEffect } from 'react';

import { Box, Typography, Container, CircularProgress, Alert } from '@mui/material';

import axios from 'axios';

const API_BASE_URL = process.env.NEXT_PUBLIC_API_URI || 'http://localhost:3024';

interface PublicPageProps {
  pageType: string;
}

export default function PublicPage({ pageType }: PublicPageProps) {
  const [page, setPage] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    fetchPage();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [pageType]);

  const fetchPage = async () => {
    try {
      setLoading(true);
      setError('');
      
      const response = await axios.get(`${API_BASE_URL}/pages?pageType=${pageType}`);
      
      if (response.data.success && response.data.data) {
        setPage(response.data.data);
      } else {
        setError('Page not found');
      }
    } catch (err: any) {
      console.error('Error fetching page:', err);
      setError(err.response?.data?.message || 'Failed to load page');
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <Container maxWidth="lg" sx={{ py: 8, display: 'flex', justifyContent: 'center' }}>
        <CircularProgress />
      </Container>
    );
  }

  if (error || !page) {
    return (
      <Container maxWidth="lg" sx={{ py: 8 }}>
        <Alert severity="error">{error || 'Page not found'}</Alert>
      </Container>
    );
  }

  if (!page.isActive) {
    return (
      <Container maxWidth="lg" sx={{ py: 8 }}>
        <Alert severity="info">This page is currently unavailable.</Alert>
      </Container>
    );
  }

  return (
    <Container maxWidth="lg" sx={{ py: 8 }}>
      <Typography variant="h3" component="h1" gutterBottom>
        {page.title}
      </Typography>
      
      <Typography variant="caption" color="textSecondary" display="block" sx={{ mb: 4 }}>
        Last updated: {new Date(page.updatedAt).toLocaleDateString()}
      </Typography>

      <Box
        sx={{
          '& h1': { fontSize: '2rem', fontWeight: 'bold', mt: 3, mb: 2 },
          '& h2': { fontSize: '1.75rem', fontWeight: 'bold', mt: 3, mb: 2 },
          '& h3': { fontSize: '1.5rem', fontWeight: 'bold', mt: 2, mb: 1 },
          '& p': { mb: 2, lineHeight: 1.7 },
          '& ul, & ol': { mb: 2, pl: 3 },
          '& li': { mb: 1 },
          '& a': { color: 'primary.main', textDecoration: 'underline' },
          '& table': { 
            width: '100%', 
            borderCollapse: 'collapse', 
            mb: 2,
            '& td, & th': { 
              border: '1px solid #ddd', 
              padding: '8px' 
            },
            '& th': { 
              backgroundColor: '#f5f5f5',
              fontWeight: 'bold'
            }
          },
        }}
        dangerouslySetInnerHTML={{ __html: page.content }}
      />
    </Container>
  );
}

