'use client';

import { useEffect, useState } from 'react';

import {
  Add as AddIcon,
  Delete as DeleteIcon,
  Edit as EditIcon,
  Search as SearchIcon
} from '@mui/icons-material';
import {
  Box,
  Button,
  Card,
  CardContent,
  Dialog,
  DialogActions,
  DialogContent,
  DialogTitle,
  IconButton,
  InputAdornment,
  Pagination,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  TextField,
  Typography
} from '@mui/material';

import useAPI from '@/hooks/useAPI';

interface BoxingClub {
  id: string;
  title: string;
  latitude?: number;
  longitude?: number;
  createdAt: string;
  updatedAt: string;
}

const PAGE_SIZE = 10;

const BoxingClubManagement = () => {
  const [boxingClubs, setBoxingClubs] = useState<BoxingClub[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [search, setSearch] = useState('');
  const [dialogOpen, setDialogOpen] = useState(false);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [currentClub, setCurrentClub] = useState<BoxingClub | null>(null);
  const [title, setTitle] = useState('');
  const [error, setError] = useState('');
  const [latitude, setLatitude] = useState<string>('');
  const [longitude, setLongitude] = useState<string>('');
  const [saving, setSaving] = useState(false);

  const fetchClubs = async () => {
    try {
      setLoading(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      const response = await api.boxingClub.getAll({
        page,
        limit: PAGE_SIZE,
        search: search.trim() || undefined
      });

      if (response.data.success) {
        setBoxingClubs(response.data.data);
        setTotalPages(response.data.pagination?.totalPages || 1);
      }
    } catch (err) {
      console.error('Failed to load boxing clubs', err);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchClubs();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [page, search]);

  const handleOpenDialog = (club?: BoxingClub) => {
    setCurrentClub(club || null);
    setTitle(club?.title || '');
    setLatitude(
      club?.latitude !== undefined && club?.latitude !== null
        ? club.latitude.toString()
        : ''
    );
    setLongitude(
      club?.longitude !== undefined && club?.longitude !== null
        ? club.longitude.toString()
        : ''
    );
    setError('');
    setDialogOpen(true);
  };

  const handleCloseDialog = () => {
    setDialogOpen(false);
    setTitle('');
    setLatitude('');
    setLongitude('');
    setCurrentClub(null);
    setSaving(false);
    setError('');
  };

  const handleSave = async () => {
    if (!title.trim()) {
      setError('Title is required');
      return;
    }

    try {
      setSaving(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();

      const payload: any = {
        title: title.trim(),
      };

      if (latitude.trim()) payload.latitude = latitude.trim();
      if (longitude.trim()) payload.longitude = longitude.trim();

      if (currentClub) {
        await api.boxingClub.update(currentClub.id, payload);
      } else {
        await api.boxingClub.create(payload);
        setPage(1);
      }

      handleCloseDialog();
      fetchClubs();
    } catch (err: any) {
      console.error('Failed to save boxing club', err);
      setError(err.response?.data?.message || 'Failed to save boxing club');
    } finally {
      setSaving(false);
    }
  };

  const handleConfirmDelete = (club: BoxingClub) => {
    setCurrentClub(club);
    setDeleteDialogOpen(true);
  };

  const handleDelete = async () => {
    if (!currentClub) return;

    try {
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      await api.boxingClub.delete(currentClub.id);
      setDeleteDialogOpen(false);
      setCurrentClub(null);
      fetchClubs();
    } catch (err) {
      console.error('Failed to delete boxing club', err);
    }
  };

  return (
    <Card>
      <CardContent>
        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3 }}>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', flexWrap: 'wrap', gap: 2 }}>
            <Typography variant='h4'>🥊 Boxing Club Management</Typography>
            <Button variant='contained' startIcon={<AddIcon />} onClick={() => handleOpenDialog()}>
              Add Boxing Club
            </Button>
          </Box>

          <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
            <TextField
              placeholder='Search boxing clubs...'
              value={search}
              onChange={event => {
                setSearch(event.target.value);
                setPage(1);
              }}
              InputProps={{
                startAdornment: (
                  <InputAdornment position='start'>
                    <SearchIcon />
                  </InputAdornment>
                )
              }}
              sx={{ flex: 1, minWidth: 240 }}
            />
          </Box>

          <TableContainer>
            <Table>
              <TableHead>
                <TableRow>
                  <TableCell>Title</TableCell>
                  <TableCell>Coordinates</TableCell>
                  <TableCell>Created</TableCell>
                  <TableCell>Updated</TableCell>
                  <TableCell align='right'>Actions</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {loading ? (
                  <TableRow>
                    <TableCell colSpan={4} align='center'>
                      Loading...
                    </TableCell>
                  </TableRow>
                ) : boxingClubs.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={4} align='center'>
                      No boxing clubs found
                    </TableCell>
                  </TableRow>
                ) : (
                  boxingClubs.map(club => (
                    <TableRow key={club.id}>
                      <TableCell>
                        <Typography variant='subtitle1'>{club.title}</Typography>
                      </TableCell>
                      <TableCell>
                        {typeof club.latitude === 'number' && typeof club.longitude === 'number'
                          ? `${club.latitude.toFixed(6)}° N, ${club.longitude.toFixed(6)}° E`
                          : 'Not set'}
                      </TableCell>
                      <TableCell>{club.createdAt ? new Date(club.createdAt).toLocaleDateString() : '-'}</TableCell>
                      <TableCell>{club.updatedAt ? new Date(club.updatedAt).toLocaleDateString() : '-'}</TableCell>
                      <TableCell align='right'>
                        <IconButton size='small' onClick={() => handleOpenDialog(club)}>
                          <EditIcon fontSize='small' />
                        </IconButton>
                        <IconButton size='small' color='error' onClick={() => handleConfirmDelete(club)}>
                          <DeleteIcon fontSize='small' />
                        </IconButton>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </TableContainer>

          {totalPages > 1 && (
            <Box sx={{ display: 'flex', justifyContent: 'center' }}>
              <Pagination count={totalPages} page={page} onChange={(_, value) => setPage(value)} color='primary' />
            </Box>
          )}
        </Box>
      </CardContent>

      <Dialog open={dialogOpen} onClose={handleCloseDialog} fullWidth maxWidth='xs'>
        <DialogTitle>{currentClub ? 'Edit Boxing Club' : 'Add Boxing Club'}</DialogTitle>
        <DialogContent sx={{ pt: 2 }}>
          <TextField
            label='Title'
            value={title}
            onChange={event => {
              setTitle(event.target.value);
              setError('');
            }}
            fullWidth
            required
            autoFocus
            error={Boolean(error)}
            helperText={error || ' '}
          />
          <TextField
            label='Latitude (-90 to 90)'
            value={latitude}
            onChange={event => setLatitude(event.target.value)}
            fullWidth
            sx={{ mt: 2 }}
            placeholder='e.g. 19.07609'
          />
          <TextField
            label='Longitude (-180 to 180)'
            value={longitude}
            onChange={event => setLongitude(event.target.value)}
            fullWidth
            sx={{ mt: 2 }}
            placeholder='e.g. 72.877426'
          />
        </DialogContent>
        <DialogActions>
          <Button onClick={handleCloseDialog}>Cancel</Button>
          <Button onClick={handleSave} variant='contained' disabled={saving}>
            {currentClub ? 'Update' : 'Create'}
          </Button>
        </DialogActions>
      </Dialog>

      <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
        <DialogTitle>Delete Boxing Club</DialogTitle>
        <DialogContent>Are you sure you want to delete &quot;{currentClub?.title}&quot;?</DialogContent>
        <DialogActions>
          <Button
            onClick={() => {
              setDeleteDialogOpen(false);
              setCurrentClub(null);
            }}
          >
            Cancel
          </Button>
          <Button color='error' variant='contained' onClick={handleDelete}>
            Delete
          </Button>
        </DialogActions>
      </Dialog>
    </Card>
  );
};

export default BoxingClubManagement;

