'use client';

import { useEffect, useState } from 'react';

import {
  Add as AddIcon,
  Delete as DeleteIcon,
  Edit as EditIcon,
  Search as SearchIcon,
} from '@mui/icons-material';
import {
  Box,
  Button,
  Card,
  CardContent,
  Dialog,
  DialogActions,
  DialogContent,
  DialogTitle,
  IconButton,
  InputAdornment,
  Pagination,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  TextField,
  Typography,
} from '@mui/material';

import useAPI from '@/hooks/useAPI';

interface Certificate {
  id: string;
  title: string;
  createdAt: string;
  updatedAt: string;
}

const PAGE_SIZE = 10;

const CertificateManagement = () => {
  const [certificates, setCertificates] = useState<Certificate[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [page, setPage] = useState<number>(1);
  const [totalPages, setTotalPages] = useState<number>(1);
  const [search, setSearch] = useState<string>('');
  const [dialogOpen, setDialogOpen] = useState<boolean>(false);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState<boolean>(false);
  const [currentCertificate, setCurrentCertificate] = useState<Certificate | null>(null);
  const [title, setTitle] = useState<string>('');
  const [error, setError] = useState<string>('');
  const [saving, setSaving] = useState<boolean>(false);

  const fetchCertificates = async () => {
    try {
      setLoading(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      const response = await api.certificate.getAll({
        page,
        limit: PAGE_SIZE,
        search: search.trim() || undefined,
      });

      if (response.data.success) {
        setCertificates(response.data.data);
        setTotalPages(response.data.pagination?.totalPages || 1);
      }
    } catch (err) {
      console.error('Failed to load certificates', err);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchCertificates();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [page, search]);

  const handleOpenDialog = (certificate?: Certificate) => {
    setCurrentCertificate(certificate || null);
    setTitle(certificate?.title || '');
    setError('');
    setDialogOpen(true);
  };

  const handleCloseDialog = () => {
    setDialogOpen(false);
    setTitle('');
    setCurrentCertificate(null);
    setSaving(false);
    setError('');
  };

  const handleSave = async () => {
    if (!title.trim()) {
      setError('Title is required');

      return;
    }

    try {
      setSaving(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();

      if (currentCertificate) {
        await api.certificate.update(currentCertificate.id, { title: title.trim() });
      } else {
        await api.certificate.create({ title: title.trim() });
        setPage(1);
      }

      handleCloseDialog();
      fetchCertificates();
    } catch (err: any) {
      console.error('Failed to save certificate', err);
      setError(err.response?.data?.message || 'Failed to save certificate');
    } finally {
      setSaving(false);
    }
  };

  const handleConfirmDelete = (certificate: Certificate) => {
    setCurrentCertificate(certificate);
    setDeleteDialogOpen(true);
  };

  const handleDelete = async () => {
    if (!currentCertificate) return;

    try {
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      await api.certificate.delete(currentCertificate.id);
      setDeleteDialogOpen(false);
      setCurrentCertificate(null);
      fetchCertificates();
    } catch (err) {
      console.error('Failed to delete certificate', err);
    }
  };

  return (
    <Card>
      <CardContent>
        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3 }}>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', flexWrap: 'wrap', gap: 2 }}>
            <Typography variant='h4'>🎖️ Certificate Management</Typography>
            <Button variant='contained' startIcon={<AddIcon />} onClick={() => handleOpenDialog()}>
              Add Certificate
            </Button>
          </Box>

          <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
            <TextField
              placeholder='Search certificates...'
              value={search}
              onChange={event => {
                setSearch(event.target.value);
                setPage(1);
              }}
              InputProps={{
                startAdornment: (
                  <InputAdornment position='start'>
                    <SearchIcon />
                  </InputAdornment>
                ),
              }}
              sx={{ flex: 1, minWidth: 240 }}
            />
          </Box>

          <TableContainer>
            <Table>
              <TableHead>
                <TableRow>
                  <TableCell>Title</TableCell>
                  <TableCell>Created</TableCell>
                  <TableCell>Updated</TableCell>
                  <TableCell align='right'>Actions</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {loading ? (
                  <TableRow>
                    <TableCell colSpan={4} align='center'>
                      Loading...
                    </TableCell>
                  </TableRow>
                ) : certificates.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={4} align='center'>
                      No certificates found
                    </TableCell>
                  </TableRow>
                ) : (
                  certificates.map(certificate => (
                    <TableRow key={certificate.id}>
                      <TableCell>
                        <Typography variant='subtitle1'>{certificate.title}</Typography>
                      </TableCell>
                      <TableCell>
                        {certificate.createdAt ? new Date(certificate.createdAt).toLocaleDateString() : '-'}
                      </TableCell>
                      <TableCell>
                        {certificate.updatedAt ? new Date(certificate.updatedAt).toLocaleDateString() : '-'}
                      </TableCell>
                      <TableCell align='right'>
                        <IconButton size='small' onClick={() => handleOpenDialog(certificate)}>
                          <EditIcon fontSize='small' />
                        </IconButton>
                        <IconButton size='small' color='error' onClick={() => handleConfirmDelete(certificate)}>
                          <DeleteIcon fontSize='small' />
                        </IconButton>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </TableContainer>

          {totalPages > 1 && (
            <Box sx={{ display: 'flex', justifyContent: 'center' }}>
              <Pagination count={totalPages} page={page} onChange={(_, value) => setPage(value)} color='primary' />
            </Box>
          )}
        </Box>
      </CardContent>

      <Dialog open={dialogOpen} onClose={handleCloseDialog} fullWidth maxWidth='xs'>
        <DialogTitle>{currentCertificate ? 'Edit Certificate' : 'Add Certificate'}</DialogTitle>
        <DialogContent sx={{ pt: 2 }}>
          <TextField
            label='Title'
            value={title}
            onChange={event => {
              setTitle(event.target.value);
              setError('');
            }}
            fullWidth
            required
            autoFocus
            error={Boolean(error)}
            helperText={error || ' '}
          />
        </DialogContent>
        <DialogActions>
          <Button onClick={handleCloseDialog}>Cancel</Button>
          <Button onClick={handleSave} variant='contained' disabled={saving}>
            {currentCertificate ? 'Update' : 'Create'}
          </Button>
        </DialogActions>
      </Dialog>

      <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
        <DialogTitle>Delete Certificate</DialogTitle>
        <DialogContent>
          Are you sure you want to delete &quot;{currentCertificate?.title}&quot;?
        </DialogContent>
        <DialogActions>
          <Button
            onClick={() => {
              setDeleteDialogOpen(false);
              setCurrentCertificate(null);
            }}
          >
            Cancel
          </Button>
          <Button color='error' variant='contained' onClick={handleDelete}>
            Delete
          </Button>
        </DialogActions>
      </Dialog>
    </Card>
  );
};

export default CertificateManagement;

