"use client";

import { useState, useEffect } from 'react';
import { TextField, Button, Box, Paper, CircularProgress, Typography, Stack } from '@mui/material';
import { useSession } from 'next-auth/react';
import { useRouter } from 'next/navigation';

const API_BASE_URL = process.env.NEXT_PUBLIC_API_URI || 'http://localhost:3024';

export default function EventForm({ mode = 'add', eventId }: { mode?: 'add'|'edit'; eventId?: string }) {
  const { data: session } = useSession();
  const router = useRouter();
  const [title, setTitle] = useState('');
  const [imageUrl, setImageUrl] = useState('');
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [previewSrc, setPreviewSrc] = useState<string>('');
  const [uploading, setUploading] = useState(false);
  const [startAt, setStartAt] = useState('');
  const [endAt, setEndAt] = useState('');
  const [category, setCategory] = useState('');

  useEffect(() => {
    if (mode === 'edit' && eventId) {
      (async () => {
        const token = session?.user?.sessionToken;
        const res = await fetch(`${API_BASE_URL}/events/${eventId}`, {
          headers: { Authorization: `Bearer ${token}`, 'Refresh-Token': session?.user?.refreshToken || '' },
        });
        const data = await res.json();
        if (data.success) {
          const ev = data.data;
          setTitle(ev.title || '');
          setImageUrl(ev.imageUrl || '');
          setPreviewSrc(ev.imageUrl || '');
          setStartAt(ev.startAt ? new Date(ev.startAt).toISOString().slice(0,16) : '');
          setEndAt(ev.endAt ? new Date(ev.endAt).toISOString().slice(0,16) : '');
          setCategory(ev.category || '');
        }
      })();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [mode, eventId, session]);

  const handleSubmit = async (e: any) => {
    e.preventDefault();
    const token = session?.user?.sessionToken;
    // Build FormData and include file if selected. The server will upload the file and set imageUrl.
    try {
      const fd = new FormData();
      fd.append('title', title);
      fd.append('startAt', startAt);
      fd.append('endAt', endAt);
      fd.append('category', category || '');
      if (selectedFile) fd.append('file', selectedFile);

      const res = await fetch(mode === 'add' ? `${API_BASE_URL}/events` : `${API_BASE_URL}/events/${eventId}`, {
        method: mode === 'add' ? 'POST' : 'PUT',
        headers: {
          Authorization: `Bearer ${token}`,
          'Refresh-Token': session?.user?.refreshToken || '',
        },
        body: fd,
      });

      const data = await res.json();
      if (data.success) {
        router.push('/admin/events');
      } else {
        alert(data.message || 'Error');
      }
    } catch (error) {
      console.error(error);
      alert('Error submitting');
    }
  };

  // Upload file to server and set imageUrl to returned location
  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0] ?? null;
    setSelectedFile(file);
    if (!file) {
      setPreviewSrc('');
      return;
    }

    // Local preview only; actual upload will happen on form submit
    const localUrl = URL.createObjectURL(file);
    setPreviewSrc(localUrl);
    setImageUrl('');
  };

  return (
    <Paper sx={{ p: 10, width: '100%' }}>
      <Box component="form" onSubmit={handleSubmit}>
        <Stack spacing={3}>
          <Typography variant="h4">{mode === 'add' ? 'Add Event' : 'Edit Event'}</Typography>

          <TextField label="Event Title" value={title} onChange={e => setTitle(e.target.value)} required fullWidth />

          <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
            <TextField
              label="Start date and time"
              type="datetime-local"
              value={startAt}
              onChange={e => setStartAt(e.target.value)}
              InputLabelProps={{ shrink: true }}
              required
              sx={{ flex: '1 1 300px' }}
            />
            <TextField
              label="End date and time"
              type="datetime-local"
              value={endAt}
              onChange={e => setEndAt(e.target.value)}
              InputLabelProps={{ shrink: true }}
              required
              sx={{ flex: '1 1 300px' }}
            />
          </Box>

          <TextField label="Event Category" value={category} onChange={e => setCategory(e.target.value)} fullWidth />

          <Box>
            <Typography variant="subtitle1" sx={{ mb: 1 }}>Event Image</Typography>
            <input
              id="event-image-file"
              type="file"
              accept="image/*"
              onChange={handleFileChange}
              style={{ display: 'none' }}
            />
            <label htmlFor="event-image-file">
              <Button component="span" variant="outlined">Choose Image</Button>
            </label>
            {uploading && <CircularProgress size={20} sx={{ ml: 2 }} />}

            {previewSrc ? (
              <Box sx={{ mt: 2 }}>
                <img src={previewSrc} alt="Preview" style={{ width: '20%', maxHeight: 420, objectFit: 'cover', borderRadius: 6 }} />
              </Box>
            ) : (
              <Box sx={{ width: '20%', mt: 2, border: '1px dashed rgba(0,0,0,0.12)', minHeight: 180, borderRadius: 2, display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
                <Typography color="text.secondary">No image selected</Typography>
              </Box>
            )}

            {selectedFile && (
              <Typography variant="caption" sx={{ display: 'block', mt: 1 }}>Selected: {selectedFile.name}</Typography>
            )}
          </Box>

          <Box sx={{ display: 'flex', gap: 2 }}>
            <Button type="submit" variant="contained" color="primary">{mode === 'add' ? 'Create' : 'Update'}</Button>
            <Button variant="outlined" onClick={() => router.push('/admin/events')}>Cancel</Button>
          </Box>
        </Stack>
      </Box>
    </Paper>
  );
}
