"use client";

import { useState, useEffect } from 'react';
import { useSession } from 'next-auth/react';
import {
  Card,
  CardContent,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  IconButton,
  Button,
  Box,
  Pagination,
} from '@mui/material';
import { Delete as DeleteIcon, Edit as EditIcon } from '@mui/icons-material';
import { useRouter } from 'next/navigation';

const API_BASE_URL = process.env.NEXT_PUBLIC_API_URI || 'http://localhost:3024';

export default function EventsManagement() {
  const { data: session } = useSession();
  const [events, setEvents] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const router = useRouter();

  const fetchEvents = async () => {
    try {
      setLoading(true);
      const token = session?.user?.sessionToken;
      const res = await fetch(`${API_BASE_URL}/events?page=${page}&limit=10`, {
        headers: {
          Authorization: `Bearer ${token}`,
          'Refresh-Token': session?.user?.refreshToken || '',
        },
      });
      const data = await res.json();
      if (data.success) {
        setEvents(data.data || []);
        setTotalPages(data.pagination?.totalPages || 1);
      } else {
        console.error('Failed to fetch events', data);
      }
    } catch (error) {
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (session?.user?.sessionToken) fetchEvents();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [page, session]);

  const handleDelete = async (id: string) => {
    if (!confirm('Delete this event?')) return;
    try {
      const token = session?.user?.sessionToken;
      const res = await fetch(`${API_BASE_URL}/events/${id}`, {
        method: 'DELETE',
        headers: {
          Authorization: `Bearer ${token}`,
          'Refresh-Token': session?.user?.refreshToken || '',
        },
      });
      const data = await res.json();
      if (data.success) fetchEvents();
      else alert(data.message || 'Failed to delete event');
    } catch (error) {
      console.error(error);
      alert('Failed to delete event');
    }
  };

  return (
    <Card>
      <CardContent>
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
          <Typography variant="h4">Events Management</Typography>
          <Button variant="contained" color="primary" onClick={() => router.push('/admin/event/add')}>Add Event</Button>
        </Box>

        {loading ? (
          <Typography>Loading...</Typography>
        ) : (
          <>
            <TableContainer component={Paper} sx={{ mt: 3 }}>
              <Table>
                <TableHead>
                  <TableRow>
                    <TableCell><strong>Image</strong></TableCell>
                    <TableCell><strong>Title</strong></TableCell>
                    <TableCell><strong>Start</strong></TableCell>
                    <TableCell><strong>End</strong></TableCell>
                    <TableCell><strong>Category</strong></TableCell>
                    <TableCell align="center"><strong>Actions</strong></TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {events.map(ev => (
                    <TableRow key={ev.id} hover>
                      <TableCell>
                        {ev.imageUrl ? <img src={ev.imageUrl} alt={ev.title} style={{ width: 80, objectFit: 'cover' }} /> : '—'}
                      </TableCell>
                      <TableCell>{ev.title}</TableCell>
                      <TableCell>{new Date(ev.startAt).toLocaleString()}</TableCell>
                      <TableCell>{new Date(ev.endAt).toLocaleString()}</TableCell>
                      <TableCell>{ev.category || '—'}</TableCell>
                      <TableCell align="center">
                        <IconButton size="small" color="primary" onClick={() => router.push(`/admin/event/edit/${ev.id}`)}>
                          <EditIcon />
                        </IconButton>
                        <IconButton size="small" color="error" onClick={() => handleDelete(ev.id)}>
                          <DeleteIcon />
                        </IconButton>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </TableContainer>

            <Box sx={{ display: 'flex', justifyContent: 'center', mt: 3 }}>
              <Pagination count={totalPages} page={page} onChange={(_, v) => setPage(v)} />
            </Box>
          </>
        )}
      </CardContent>
    </Card>
  );
}
