'use client';

import { ChangeEvent, useEffect, useMemo, useState } from 'react';

import {
  Box,
  Button,
  Card,
  CardContent,
  Dialog,
  DialogActions,
  DialogContent,
  DialogTitle,
  FormControlLabel,
  IconButton,
  Switch,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  TextField,
  Typography,
  Alert,
} from '@mui/material';
import {
  Add as AddIcon,
  Edit as EditIcon,
  Delete as DeleteIcon,
  Image as ImageIcon,
} from '@mui/icons-material';

import useAPI from '@/hooks/useAPI';

type HomeSection = {
  id: string;
  title: string;
  imageUrl: string;
  order: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
};

type FormState = {
  id?: string;
  title: string;
  order: number;
  isActive: boolean;
  file: File | null;
  previewUrl: string;
  existingImageUrl: string;
};

const getInitialFormState = (): FormState => ({
  title: '',
  order: 0,
  isActive: true,
  file: null,
  previewUrl: '',
  existingImageUrl: '',
});

const HomeManagement = () => {
  const [sections, setSections] = useState<HomeSection[]>([]);
  const [loading, setLoading] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [formState, setFormState] = useState<FormState>(getInitialFormState());
  const [selectedSection, setSelectedSection] = useState<HomeSection | null>(null);
  const [formError, setFormError] = useState<string>('');

  const isEditing = useMemo(() => Boolean(formState.id), [formState.id]);

  const fetchSections = async () => {
    try {
      setLoading(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      const response = await api.homeManagement.get({ includeInactive: true });
      if (response.data.success) {
        setSections(response.data.data || []);
      }
    } catch (error) {
      console.error('Failed to fetch home sections:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchSections();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    return () => {
      if (formState.previewUrl?.startsWith('blob:')) {
        URL.revokeObjectURL(formState.previewUrl);
      }
    };
  }, [formState.previewUrl]);

  const handleOpenDialog = (section?: HomeSection) => {
    if (section) {
      setFormState({
        id: section.id,
        title: section.title,
        order: section.order ?? 0,
        isActive: section.isActive,
        file: null,
        previewUrl: '',
        existingImageUrl: section.imageUrl,
      });
    } else {
      setFormState(getInitialFormState());
    }
    setFormError('');
    setDialogOpen(true);
  };

  const handleCloseDialog = () => {
    setDialogOpen(false);
    setFormState(getInitialFormState());
    setFormError('');
  };

  const handleFileChange = (event: ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    const previewUrl = URL.createObjectURL(file);
    setFormState(prev => ({
      ...prev,
      file,
      previewUrl,
    }));
  };

  const handleSubmit = async () => {
    if (!formState.title.trim()) {
      setFormError('Title is required');
      return;
    }

    if (!isEditing && !formState.file) {
      setFormError('Please upload an image');
      return;
    }

    try {
      setSubmitting(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      const payload = new FormData();
      payload.append('title', formState.title.trim());
      payload.append('order', String(formState.order ?? 0));
      payload.append('isActive', String(formState.isActive));
      if (formState.file) {
        payload.append('file', formState.file);
      }

      if (isEditing && formState.id) {
        await api.homeManagement.update(formState.id, payload);
      } else {
        await api.homeManagement.create(payload);
      }

      handleCloseDialog();
      fetchSections();
    } catch (error) {
      console.error('Failed to save home section:', error);
      setFormError('Failed to save home section. Please try again.');
    } finally {
      setSubmitting(false);
    }
  };

  const handleDelete = async () => {
    if (!selectedSection) return;
    try {
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      await api.homeManagement.delete(selectedSection.id);
      setDeleteDialogOpen(false);
      setSelectedSection(null);
      fetchSections();
    } catch (error) {
      console.error('Failed to delete home section:', error);
    }
  };

  return (
    <Card>
      <CardContent>
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
          <Box>
            <Typography variant="h4">Home Page Management</Typography>
            <Typography variant="body2" color="text.secondary">
              Manage the blocks that appear on the public home page. Each block needs a title and an image.
            </Typography>
          </Box>
          <Button variant="contained" startIcon={<AddIcon />} onClick={() => handleOpenDialog()}>
            Add Section
          </Button>
        </Box>

        <TableContainer>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>Preview</TableCell>
                <TableCell>Title</TableCell>
                <TableCell>Status</TableCell>
                <TableCell>Order</TableCell>
                <TableCell>Updated</TableCell>
                <TableCell align="right">Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {loading ? (
                <TableRow>
                  <TableCell colSpan={6} align="center">
                    Loading...
                  </TableCell>
                </TableRow>
              ) : sections.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={6} align="center">
                    No home sections created yet.
                  </TableCell>
                </TableRow>
              ) : (
                sections.map(section => (
                  <TableRow key={section.id} hover>
                    <TableCell>
                      {section.imageUrl ? (
                        <img
                          src={section.imageUrl}
                          alt={section.title}
                          style={{ width: 80, height: 80, objectFit: 'cover', borderRadius: 8 }}
                        />
                      ) : (
                        <Box
                          sx={{
                            width: 80,
                            height: 80,
                            borderRadius: 2,
                            bgcolor: 'action.hover',
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'center',
                            color: 'text.secondary',
                          }}
                        >
                          <ImageIcon />
                        </Box>
                      )}
                    </TableCell>
                    <TableCell>
                      <Typography variant="subtitle1">{section.title}</Typography>
                    </TableCell>
                    <TableCell>
                      <Typography color={section.isActive ? 'success.main' : 'text.disabled'}>
                        {section.isActive ? 'Active' : 'Hidden'}
                      </Typography>
                    </TableCell>
                    <TableCell>{section.order ?? 0}</TableCell>
                    <TableCell>{new Date(section.updatedAt).toLocaleString()}</TableCell>
                    <TableCell align="right">
                      <IconButton color="primary" onClick={() => handleOpenDialog(section)}>
                        <EditIcon />
                      </IconButton>
                      <IconButton
                        color="error"
                        onClick={() => {
                          setSelectedSection(section);
                          setDeleteDialogOpen(true);
                        }}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </TableContainer>
      </CardContent>

      {/* Create / Update dialog */}
      <Dialog open={dialogOpen} onClose={handleCloseDialog} fullWidth maxWidth="sm">
        <DialogTitle>{isEditing ? 'Edit Home Section' : 'Add Home Section'}</DialogTitle>
        <DialogContent dividers>
          {formError && (
            <Alert severity="error" sx={{ mb: 2 }}>
              {formError}
            </Alert>
          )}
          <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3 }}>
            <TextField
              label="Title"
              value={formState.title}
              onChange={e => setFormState(prev => ({ ...prev, title: e.target.value }))}
              fullWidth
              required
            />
            <TextField
              label="Display order"
              type="number"
              value={formState.order}
              onChange={e => setFormState(prev => ({ ...prev, order: Number(e.target.value) }))}
              fullWidth
              helperText="Lower numbers appear first on the home page"
            />
            <FormControlLabel
              control={
                <Switch
                  checked={formState.isActive}
                  onChange={e => setFormState(prev => ({ ...prev, isActive: e.target.checked }))}
                />
              }
              label={formState.isActive ? 'Visible on home page' : 'Hidden from home page'}
            />
            <Button variant="outlined" component="label">
              Upload Image
              <input type="file" hidden accept="image/*" onChange={handleFileChange} />
            </Button>
            {(formState.previewUrl || formState.existingImageUrl) && (
              <Box>
                <Typography variant="caption" color="text.secondary">
                  Preview
                </Typography>
                <Box
                  component="img"
                  src={formState.previewUrl || formState.existingImageUrl}
                  alt="Preview"
                  sx={{
                    width: '100%',
                    maxHeight: 240,
                    objectFit: 'cover',
                    borderRadius: 2,
                    mt: 1,
                  }}
                />
              </Box>
            )}
          </Box>
        </DialogContent>
        <DialogActions>
          <Button onClick={handleCloseDialog}>Cancel</Button>
          <Button onClick={handleSubmit} variant="contained" disabled={submitting}>
            {submitting ? 'Saving...' : isEditing ? 'Update' : 'Create'}
          </Button>
        </DialogActions>
      </Dialog>

      {/* Delete dialog */}
      <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
        <DialogTitle>Delete Home Section</DialogTitle>
        <DialogContent>
          Are you sure you want to delete &quot;{selectedSection?.title}&quot;?
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setDeleteDialogOpen(false)}>Cancel</Button>
          <Button color="error" variant="contained" onClick={handleDelete}>
            Delete
          </Button>
        </DialogActions>
      </Dialog>
    </Card>
  );
};

export default HomeManagement;

