'use client';

import { useState, useEffect } from 'react';

import { useRouter } from 'next/navigation';
import dynamic from 'next/dynamic';

import {
  Card,
  CardContent,
  Button,
  TextField,
  Box,
  Typography,
  Switch,
  FormControlLabel,
  Alert,
  CircularProgress,
} from '@mui/material';

import useAPI from '@/hooks/useAPI';

import 'react-quill/dist/quill.snow.css';

const ReactQuill = dynamic(() => import('react-quill'), { ssr: false });

const modules = {
  toolbar: [
    [{ 'header': [1, 2, 3, false] }],
    ['bold', 'italic', 'underline', 'strike'],
    [{ 'list': 'ordered'}, { 'list': 'bullet' }],
    ['blockquote', 'code-block'],
    ['link', 'image'],
    ['clean']
  ],
};

interface NewsEditorProps {
  newsId?: string;
}

export default function NewsEditor({ newsId }: NewsEditorProps) {
  const router = useRouter();
  const [title, setTitle] = useState('');
  const [excerpt, setExcerpt] = useState('');
  const [content, setContent] = useState('');
  const [imageUrl, setImageUrl] = useState('');
  const [isPublished, setIsPublished] = useState(false);
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [message, setMessage] = useState({ type: '', text: '' });

  const fetchNews = async () => {
    try {
      setLoading(true);
      setMessage({ type: '', text: '' });
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      const response = await api.news.getById(newsId!);
      
      if (response.data.success) {
        const data = response.data.data;
        setTitle(data.title);
        setExcerpt(data.excerpt || '');
        setContent(data.content);
        setImageUrl(data.imageUrl || '');
        setIsPublished(data.isPublished);
      } else {
        setMessage({ type: 'error', text: 'News not found' });
      }
    } catch (error: any) {
      console.error('Error fetching news:', error);
      setMessage({ 
        type: 'error', 
        text: error.response?.data?.message || 'Failed to load news. Please try again.' 
      });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (newsId) {
      fetchNews();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [newsId]);

  const handleSave = async () => {
    if (!title.trim() || !content.trim()) {
      setMessage({ type: 'error', text: 'Title and content are required' });

      return;
    }

    try {
      setSaving(true);
      setMessage({ type: '', text: '' });
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();

      const newsData = {
        title,
        excerpt,
        content,
        imageUrl,
        isPublished,
      };

      if (newsId) {
        await api.news.update(newsId, newsData);
        setMessage({ type: 'success', text: 'News updated successfully!' });
      } else {
        await api.news.create(newsData);
        setMessage({ type: 'success', text: 'News created successfully!' });
        setTimeout(() => router.push('/admin/news'), 1500);
      }
    } catch (error: any) {
      console.error('Error saving news:', error);
      setMessage({ type: 'error', text: error.response?.data?.message || 'Failed to save news' });
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <Box sx={{ display: 'flex', justifyContent: 'center', p: 4 }}>
        <CircularProgress />
      </Box>
    );
  }

  return (
    <Card>
      <CardContent>
        <Typography variant="h4" gutterBottom>
          {newsId ? '✏️ Edit News' : '➕ Create News'}
        </Typography>

        {message.text && (
          <Alert severity={message.type as any} sx={{ mb: 2 }}>
            {message.text}
          </Alert>
        )}

        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3, mt: 3 }}>
          <TextField
            label="Title *"
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            fullWidth
            required
          />

          <TextField
            label="Excerpt (Short Description)"
            value={excerpt}
            onChange={(e) => setExcerpt(e.target.value)}
            fullWidth
            multiline
            rows={2}
            placeholder="Brief summary of the news..."
          />

          <TextField
            label="Featured Image URL"
            value={imageUrl}
            onChange={(e) => setImageUrl(e.target.value)}
            fullWidth
            placeholder="https://example.com/image.jpg"
          />

          <FormControlLabel
            control={
              <Switch
                checked={isPublished}
                onChange={(e) => setIsPublished(e.target.checked)}
                color="primary"
              />
            }
            label={isPublished ? "Published (visible to users)" : "Draft (not visible)"}
          />

          <Box>
            <Typography variant="subtitle1" gutterBottom>
              Content *
            </Typography>
            {typeof window !== 'undefined' && (
              <Box sx={{ 
                '& .ql-container': { minHeight: '400px' }
              }}>
                <ReactQuill
                  theme="snow"
                  value={content}
                  onChange={setContent}
                  modules={modules}
                  placeholder="Write your news content here..."
                />
              </Box>
            )}
          </Box>

          <Box sx={{ display: 'flex', gap: 2, justifyContent: 'flex-end' }}>
            <Button
              variant="outlined"
              onClick={() => router.push('/admin/news')}
            >
              Cancel
            </Button>
            <Button
              variant="contained"
              onClick={handleSave}
              disabled={saving || !title.trim() || !content.trim()}
            >
              {saving ? <CircularProgress size={24} /> : (newsId ? 'Update' : 'Create')}
            </Button>
          </Box>
        </Box>
      </CardContent>
    </Card>
  );
}

