'use client';

import { useState, useEffect } from 'react';

import dynamic from 'next/dynamic';

import { 
  Card, 
  CardContent, 
  Button, 
  TextField, 
  Box, 
  Typography,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Alert,
  CircularProgress,
  Switch,
  FormControlLabel
} from '@mui/material';

import useAPI from '@/hooks/useAPI';

import 'react-quill/dist/quill.snow.css';

// Dynamically import ReactQuill to avoid SSR issues
const ReactQuill = dynamic(() => import('react-quill'), { 
  ssr: false,
  loading: () => <Box sx={{ p: 4, textAlign: 'center' }}>Loading editor...</Box>
});

const PAGE_TYPES = [
  { value: 'PRIVACY_POLICY', label: 'Privacy Policy' },
  { value: 'TERMS_OF_USE', label: 'Terms of Use' },
  { value: 'ABOUT_US', label: 'About Us' },
  { value: 'CONTACT_US', label: 'Contact Us' },
];

// Quill editor modules configuration
const modules = {
  toolbar: [
    [{ 'header': [1, 2, 3, 4, 5, 6, false] }],
    [{ 'font': [] }],
    [{ 'size': ['small', false, 'large', 'huge'] }],
    ['bold', 'italic', 'underline', 'strike'],
    [{ 'color': [] }, { 'background': [] }],
    [{ 'script': 'sub'}, { 'script': 'super' }],
    [{ 'list': 'ordered'}, { 'list': 'bullet' }],
    [{ 'indent': '-1'}, { 'indent': '+1' }],
    [{ 'align': [] }],
    ['blockquote', 'code-block'],
    ['link', 'image', 'video'],
    ['clean']
  ],
};

const formats = [
  'header', 'font', 'size',
  'bold', 'italic', 'underline', 'strike',
  'color', 'background',
  'script',
  'list', 'bullet',
  'indent', 'align',
  'blockquote', 'code-block',
  'link', 'image', 'video'
];

export default function PageEditor() {
  const [pageType, setPageType] = useState('PRIVACY_POLICY');
  const [title, setTitle] = useState('');
  const [content, setContent] = useState('');
  const [isActive, setIsActive] = useState(true);
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [message, setMessage] = useState({ type: '', text: '' });
  const [existingPages, setExistingPages] = useState<any[]>([]);

  const fetchAllPages = async () => {
    try {
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      const response = await api.pages.get();
      if (response.data.success) {
        setExistingPages(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching pages:', error);
    }
  };

  const fetchPage = async () => {
    try {
      setLoading(true);
      setMessage({ type: '', text: '' });
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      
      const response = await api.pages.get({ pageType });
      
      if (response.data.success && response.data.data) {
        const page = response.data.data;
        setTitle(page.title);
        setContent(page.content);
        setIsActive(page.isActive);
      } else {
        // No existing page, reset form
        setTitle(PAGE_TYPES.find(p => p.value === pageType)?.label || '');
        setContent('');
        setIsActive(true);
      }
    } catch (error: any) {
      if (error.response?.status === 404) {
        // No existing page, reset form
        setTitle(PAGE_TYPES.find(p => p.value === pageType)?.label || '');
        setContent('');
        setIsActive(true);
      } else {
        console.error('Error fetching page:', error);
        setMessage({ type: 'error', text: 'Failed to load page' });
      }
    } finally {
      setLoading(false);
    }
  };

  // Fetch existing page when page type changes
  useEffect(() => {
    fetchPage();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [pageType]);

  // Fetch all pages on mount
  useEffect(() => {
    fetchAllPages();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const handleSave = async () => {
    if (!title.trim() || !content.trim()) {
      setMessage({ type: 'error', text: 'Title and content are required' });

      return;
    }

    try {
      setSaving(true);
      setMessage({ type: '', text: '' });
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();

      const response = await api.pages.upsert({
        pageType,
        title,
        content,
        isActive,
      });

      if (response.data.success) {
        setMessage({ 
          type: 'success', 
          text: `${title} saved successfully!` 
        });
        fetchAllPages(); // Refresh the list
      } else {
        setMessage({ type: 'error', text: response.data.message || 'Failed to save page' });
      }
    } catch (error: any) {
      console.error('Error saving page:', error);
      setMessage({ 
        type: 'error', 
        text: error.response?.data?.message || 'Failed to save page' 
      });
    } finally {
      setSaving(false);
    }
  };

  return (
    <Card>
      <CardContent>
        <Typography variant="h4" gutterBottom>
          📝 Page Content Editor
        </Typography>
        <Typography variant="body2" color="textSecondary" gutterBottom>
          Create and edit Privacy Policy, Terms of Use, and other important pages with rich text formatting
        </Typography>

        {message.text && (
          <Alert severity={message.type as any} sx={{ mb: 2, mt: 2 }}>
            {message.text}
          </Alert>
        )}

        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3, mt: 3 }}>
          {/* Page Type Selector */}
          <FormControl fullWidth>
            <InputLabel>Page Type</InputLabel>
            <Select
              value={pageType}
              onChange={(e) => setPageType(e.target.value)}
              label="Page Type"
            >
              {PAGE_TYPES.map((type) => (
                <MenuItem key={type.value} value={type.value}>
                  {type.label}
                  {existingPages.find(p => p.pageType === type.value) && ' ✓'}
                </MenuItem>
              ))}
            </Select>
          </FormControl>

          {/* Title Field */}
          <TextField
            label="Page Title"
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            fullWidth
            required
            placeholder="Enter page title..."
          />

          {/* Active Status */}
          <FormControlLabel
            control={
              <Switch
                checked={isActive}
                onChange={(e) => setIsActive(e.target.checked)}
                color="primary"
              />
            }
            label="Active (visible to users)"
          />

          {/* Rich Text Editor */}
          <Box>
            <Typography variant="subtitle1" gutterBottom>
              Content *
            </Typography>
            {loading ? (
              <Box sx={{ display: 'flex', justifyContent: 'center', p: 4 }}>
                <CircularProgress />
              </Box>
            ) : (
              <Box sx={{ 
                '& .quill': {
                  bgcolor: 'background.paper',
                  borderRadius: 1,
                  border: '1px solid',
                  borderColor: 'divider',
                },
                '& .ql-toolbar': {
                  bgcolor: 'action.hover',
                  borderTopLeftRadius: 4,
                  borderTopRightRadius: 4,
                  borderBottom: '1px solid',
                  borderColor: 'divider',
                },
                '& .ql-container': {
                  minHeight: '400px',
                  fontSize: '16px',
                  fontFamily: 'inherit',
                },
                '& .ql-editor': {
                  minHeight: '400px',
                }
              }}>
                {typeof window !== 'undefined' && (
                  <ReactQuill
                    theme="snow"
                    value={content}
                    onChange={setContent}
                    modules={modules}
                    formats={formats}
                    placeholder="Write your content here... Use the toolbar above to format text, add links, images, and more."
                  />
                )}
              </Box>
            )}
          </Box>

          {/* Action Buttons */}
          <Box sx={{ display: 'flex', gap: 2, justifyContent: 'flex-end' }}>
            <Button
              variant="outlined"
              onClick={() => fetchPage()}
              disabled={loading || saving}
            >
              Reset
            </Button>
            <Button
              variant="contained"
              color="primary"
              onClick={handleSave}
              disabled={loading || saving || !title.trim() || !content.trim()}
            >
              {saving ? <CircularProgress size={24} /> : 'Save Page'}
            </Button>
          </Box>
        </Box>

        {/* Existing Pages List */}
        {existingPages.length > 0 && (
          <Box sx={{ mt: 4 }}>
            <Typography variant="h6" gutterBottom>
              📄 Existing Pages
            </Typography>
            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1 }}>
              {existingPages.map((page) => (
                <Box
                  key={page.id}
                  sx={{
                    p: 2,
                    border: '1px solid',
                    borderColor: 'divider',
                    borderRadius: 1,
                    display: 'flex',
                    justifyContent: 'space-between',
                    alignItems: 'center',
                    cursor: 'pointer',
                    transition: 'all 0.2s',
                    '&:hover': { 
                      bgcolor: 'action.hover',
                      borderColor: 'primary.main',
                    }
                  }}
                  onClick={() => setPageType(page.pageType)}
                >
                  <Box>
                    <Typography variant="subtitle1" fontWeight="bold">
                      {page.title}
                    </Typography>
                    <Typography variant="caption" color="textSecondary">
                      {PAGE_TYPES.find(t => t.value === page.pageType)?.label} • 
                      {page.isActive ? ' 🟢 Active' : ' 🔴 Inactive'} • 
                      Updated: {new Date(page.updatedAt).toLocaleDateString()}
                    </Typography>
                  </Box>
                  <Button
                    size="small"
                    variant="outlined"
                    onClick={(e) => {
                      e.stopPropagation();
                      setPageType(page.pageType);
                    }}
                  >
                    Edit
                  </Button>
                </Box>
              ))}
            </Box>
          </Box>
        )}
      </CardContent>
    </Card>
  );
}
