'use client';

import { useEffect, useState } from 'react';

import {
  Add as AddIcon,
  Delete as DeleteIcon,
  Edit as EditIcon,
  Search as SearchIcon
} from '@mui/icons-material';
import {
  Box,
  Button,
  Card,
  CardContent,
  Dialog,
  DialogActions,
  DialogContent,
  DialogTitle,
  IconButton,
  InputAdornment,
  Pagination,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  TextField,
  Typography
} from '@mui/material';

import useAPI from '@/hooks/useAPI';

interface Professional {
  id: string;
  title: string;
  createdAt: string;
  updatedAt: string;
}

const PAGE_SIZE = 10;

const ProfessionalManagement = () => {
  const [professionals, setProfessionals] = useState<Professional[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [search, setSearch] = useState('');
  const [dialogOpen, setDialogOpen] = useState(false);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [currentProfessional, setCurrentProfessional] = useState<Professional | null>(null);
  const [title, setTitle] = useState('');
  const [error, setError] = useState('');
  const [saving, setSaving] = useState(false);

  const fetchProfessionals = async () => {
    try {
      setLoading(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      const response = await api.professional.getAll({
        page,
        limit: PAGE_SIZE,
        search: search.trim() || undefined
      });

      if (response.data.success) {
        setProfessionals(response.data.data);
        setTotalPages(response.data.pagination?.totalPages || 1);
      }
    } catch (err) {
      console.error('Failed to load professionals', err);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchProfessionals();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [page, search]);

  const handleOpenDialog = (professional?: Professional) => {
    setCurrentProfessional(professional || null);
    setTitle(professional?.title || '');
    setError('');
    setDialogOpen(true);
  };

  const handleCloseDialog = () => {
    setDialogOpen(false);
    setTitle('');
    setCurrentProfessional(null);
    setSaving(false);
    setError('');
  };

  const handleSave = async () => {
    if (!title.trim()) {
      setError('Title is required');
      return;
    }

    try {
      setSaving(true);
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();

      if (currentProfessional) {
        await api.professional.update(currentProfessional.id, { title: title.trim() });
      } else {
        await api.professional.create({ title: title.trim() });
        setPage(1);
      }

      handleCloseDialog();
      fetchProfessionals();
    } catch (err: any) {
      console.error('Failed to save professional', err);
      setError(err.response?.data?.message || 'Failed to save professional');
    } finally {
      setSaving(false);
    }
  };

  const handleConfirmDelete = (professional: Professional) => {
    setCurrentProfessional(professional);
    setDeleteDialogOpen(true);
  };

  const handleDelete = async () => {
    if (!currentProfessional) return;

    try {
      // eslint-disable-next-line react-hooks/rules-of-hooks
      const api = await useAPI();
      await api.professional.delete(currentProfessional.id);
      setDeleteDialogOpen(false);
      setCurrentProfessional(null);
      fetchProfessionals();
    } catch (err) {
      console.error('Failed to delete professional', err);
    }
  };

  return (
    <Card>
      <CardContent>
        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3 }}>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', flexWrap: 'wrap', gap: 2 }}>
            <Typography variant='h4'>👩‍⚖️ Professionals Management</Typography>
            <Button variant='contained' startIcon={<AddIcon />} onClick={() => handleOpenDialog()}>
              Add Professional
            </Button>
          </Box>

          <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
            <TextField
              placeholder='Search professionals...'
              value={search}
              onChange={event => {
                setSearch(event.target.value);
                setPage(1);
              }}
              InputProps={{
                startAdornment: (
                  <InputAdornment position='start'>
                    <SearchIcon />
                  </InputAdornment>
                )
              }}
              sx={{ flex: 1, minWidth: 240 }}
            />
          </Box>

          <TableContainer>
            <Table>
              <TableHead>
                <TableRow>
                  <TableCell>Title</TableCell>
                  <TableCell>Created</TableCell>
                  <TableCell>Updated</TableCell>
                  <TableCell align='right'>Actions</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {loading ? (
                  <TableRow>
                    <TableCell colSpan={4} align='center'>
                      Loading...
                    </TableCell>
                  </TableRow>
                ) : professionals.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={4} align='center'>
                      No professionals found
                    </TableCell>
                  </TableRow>
                ) : (
                  professionals.map(professional => (
                    <TableRow key={professional.id}>
                      <TableCell>
                        <Typography variant='subtitle1'>{professional.title}</Typography>
                      </TableCell>
                      <TableCell>
                        {professional.createdAt ? new Date(professional.createdAt).toLocaleDateString() : '-'}
                      </TableCell>
                      <TableCell>
                        {professional.updatedAt ? new Date(professional.updatedAt).toLocaleDateString() : '-'}
                      </TableCell>
                      <TableCell align='right'>
                        <IconButton size='small' onClick={() => handleOpenDialog(professional)}>
                          <EditIcon fontSize='small' />
                        </IconButton>
                        <IconButton size='small' color='error' onClick={() => handleConfirmDelete(professional)}>
                          <DeleteIcon fontSize='small' />
                        </IconButton>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </TableContainer>

          {totalPages > 1 && (
            <Box sx={{ display: 'flex', justifyContent: 'center' }}>
              <Pagination count={totalPages} page={page} onChange={(_, value) => setPage(value)} color='primary' />
            </Box>
          )}
        </Box>
      </CardContent>

      <Dialog open={dialogOpen} onClose={handleCloseDialog} fullWidth maxWidth='xs'>
        <DialogTitle>{currentProfessional ? 'Edit Professional' : 'Add Professional'}</DialogTitle>
        <DialogContent sx={{ pt: 2 }}>
          <TextField
            label='Title'
            value={title}
            onChange={event => {
              setTitle(event.target.value);
              setError('');
            }}
            fullWidth
            required
            autoFocus
            error={Boolean(error)}
            helperText={error || ' '}
          />
        </DialogContent>
        <DialogActions>
          <Button onClick={handleCloseDialog}>Cancel</Button>
          <Button onClick={handleSave} variant='contained' disabled={saving}>
            {currentProfessional ? 'Update' : 'Create'}
          </Button>
        </DialogActions>
      </Dialog>

      <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
        <DialogTitle>Delete Professional</DialogTitle>
        <DialogContent>Are you sure you want to delete &quot;{currentProfessional?.title}&quot;?</DialogContent>
        <DialogActions>
          <Button
            onClick={() => {
              setDeleteDialogOpen(false);
              setCurrentProfessional(null);
            }}
          >
            Cancel
          </Button>
          <Button color='error' variant='contained' onClick={handleDelete}>
            Delete
          </Button>
        </DialogActions>
      </Dialog>
    </Card>
  );
};

export default ProfessionalManagement;

