'use client';

import { useState, useEffect } from 'react';

import { useSession } from 'next-auth/react';

import { 
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Button, 
  TextField, 
  MenuItem, 
  Select, 
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Chip,
  IconButton,
  Box,
  Typography,
  Pagination,
  Card,
  CardContent,
  InputAdornment
} from '@mui/material';
import { 
  Delete as DeleteIcon, 
  Visibility as ViewIcon,
  Search as SearchIcon 
} from '@mui/icons-material';

// Get API URL from environment variable
const API_BASE_URL = process.env.NEXT_PUBLIC_API_URI || 'http://localhost:3024';

interface User {
  id: string;
  firstName?: string;
  lastName?: string;
  email: string;
  phoneNumber?: string;
  userType?: 'GUEST_GENERAL_VISITOR' | 'FIGHTER' | 'COACH' | 'PROFESSIONAL';
  role: 'ADMIN' | 'DEFAULT';
  city?: string;
  country?: string;
  createdAt: string;
}

interface UserDetailsDialogProps {
  open: boolean;
  onClose: () => void;
  user: any;
}

// Helper to format user type display
const formatUserType = (userType?: string) => {
  if (!userType) return undefined
  if (userType === 'GUEST_GENERAL_VISITOR') return 'GUEST'
  return userType
}

const UserDetailsDialog = ({ open, onClose, user }: UserDetailsDialogProps) => {
  if (!user) return null;

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <DialogTitle>User Details</DialogTitle>
      <DialogContent>
        <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2, mt: 2 }}>
          <Box>
            <Typography variant="caption" color="textSecondary">Name</Typography>
            <Typography variant="body1">
              {[user.firstName, user.middleName, user.lastName].filter(Boolean).join(' ') || 
                <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
              }
            </Typography>
          </Box>

          <Box>
            <Typography variant="caption" color="textSecondary">Email</Typography>
            <Typography variant="body1">{user.email}</Typography>
          </Box>

          <Box>
            <Typography variant="caption" color="textSecondary">Phone</Typography>
            <Typography variant="body1">
              {user.phoneNumber || 
                <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
              }
            </Typography>
          </Box>

          <Box>
            <Typography variant="caption" color="textSecondary">User Type</Typography>
            <Typography variant="body1">
              {formatUserType(user.userType) || 
                <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
              }
            </Typography>
          </Box>

          <Box>
            <Typography variant="caption" color="textSecondary">Role</Typography>
            <Typography variant="body1">{user.role}</Typography>
          </Box>

          <Box>
            <Typography variant="caption" color="textSecondary">Gender</Typography>
            <Typography variant="body1">
              {user.gender || 
                <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
              }
            </Typography>
          </Box>

          <Box>
            <Typography variant="caption" color="textSecondary">City</Typography>
            <Typography variant="body1">
              {user.city || 
                <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
              }
            </Typography>
          </Box>

          <Box>
            <Typography variant="caption" color="textSecondary">Country</Typography>
            <Typography variant="body1">
              {user.country || 
                <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
              }
            </Typography>
          </Box>

          {user.sportType && (
            <Box>
              <Typography variant="caption" color="textSecondary">Sport Type</Typography>
              <Typography variant="body1">{user.sportType}</Typography>
            </Box>
          )}

          {user.experience && (
            <Box>
              <Typography variant="caption" color="textSecondary">Experience</Typography>
              <Typography variant="body1">{user.experience}</Typography>
            </Box>
          )}
        </Box>

        {/* Document Links */}
        {(user.signingForms || user.identityDocument || user.medicalDocument || user.athleteCard || user.coachCertificate || user.judgesCertificate || user.experienceProof) && (
          <Box sx={{ mt: 3 }}>
            <Typography variant="h6" gutterBottom>Documents</Typography>
            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1 }}>
              {user.signingForms && (
                <Button variant="outlined" size="small" href={user.signingForms} target="_blank" rel="noopener noreferrer">
                  View Signing Forms
                </Button>
              )}
              {user.identityDocument && (
                <Button variant="outlined" size="small" href={user.identityDocument} target="_blank" rel="noopener noreferrer">
                  View Identity Document
                </Button>
              )}
              {user.medicalDocument && (
                <Button variant="outlined" size="small" href={user.medicalDocument} target="_blank" rel="noopener noreferrer">
                  View Medical Document
                </Button>
              )}
              {user.athleteCard && (
                <Button variant="outlined" size="small" href={user.athleteCard} target="_blank" rel="noopener noreferrer">
                  View Athlete Card
                </Button>
              )}
              {user.coachCertificate && (
                <Button variant="outlined" size="small" href={user.coachCertificate} target="_blank" rel="noopener noreferrer">
                  View Coach Certificate
                </Button>
              )}
              {user.judgesCertificate && (
                <Button variant="outlined" size="small" href={user.judgesCertificate} target="_blank" rel="noopener noreferrer">
                  View Judges Certificate
                </Button>
              )}
              {user.experienceProof && (
                <Button variant="outlined" size="small" href={user.experienceProof} target="_blank" rel="noopener noreferrer">
                  View Experience Proof
                </Button>
              )}
            </Box>
          </Box>
        )}
      </DialogContent>
      <DialogActions>
        <Button onClick={onClose}>Close</Button>
      </DialogActions>
    </Dialog>
  );
};

export default function UserManagement() {
  const { data: session } = useSession();
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [search, setSearch] = useState('');
  const [userTypeFilter, setUserTypeFilter] = useState('ALL');
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<User | null>(null);
  const [viewDialogOpen, setViewDialogOpen] = useState(false);
  const [selectedUserDetails, setSelectedUserDetails] = useState<any>(null);

  // Fetch users
  const fetchUsers = async () => {
    try {
      setLoading(true);
      const token = session?.user?.sessionToken;
      
      console.log('=== DEBUG INFO ===');
      console.log('Session:', session);
      console.log('Token:', token);
      console.log('User Role:', session?.user?.role);
      console.log('API URL:', `${API_BASE_URL}/user?page=${page}&limit=10&search=${search}&userType=${userTypeFilter}`);
      
      if (!token) {
        alert('No authentication token found. Please login again.');
        setLoading(false);

        return;
      }
      
      const response = await fetch(
        `${API_BASE_URL}/user?page=${page}&limit=10&search=${search}&userType=${userTypeFilter}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Refresh-Token': session?.user?.refreshToken || '',
          },
        }
      );

      const data = await response.json();
      
      console.log('Response status:', response.status);
      console.log('Response data:', data);
      
      if (response.status === 403) {
        console.error('403 Forbidden - Token:', token);
        console.error('Response:', data);
        alert('Access Denied: You are not authorized to view users. Please login again.');

        return;
      }
      
      if (data.success) {
          // Do not include ADMIN role users in the displayed list
          const visibleUsers = Array.isArray(data.data) ? data.data.filter((u: any) => u.role !== 'ADMIN') : []
          setUsers(visibleUsers);
        setTotalPages(data.pagination.totalPages);
      } else {
        console.error('API Error:', data);
        alert(data.message || 'Failed to fetch users');
      }
    } catch (error) {
      console.error('Error fetching users:', error);
      alert('Failed to fetch users. Please check your connection and try again.');
    } finally {
      setLoading(false);
    }
  };

  // Fetch user details
  const fetchUserDetails = async (userId: string) => {
    try {
      const token = session?.user?.sessionToken;
      
      const response = await fetch(`${API_BASE_URL}/user/${userId}`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Refresh-Token': session?.user?.refreshToken || '',
        },
      });

      const data = await response.json();
      
      if (data.success) {
        setSelectedUserDetails(data.data);
        setViewDialogOpen(true);
      }
    } catch (error) {
      console.error('Error fetching user details:', error);
      alert('Failed to fetch user details');
    }
  };

  // Delete user
  const handleDeleteUser = async () => {
    if (!selectedUser) return;

    try {
      const token = session?.user?.sessionToken;
      
      const response = await fetch(`${API_BASE_URL}/user/${selectedUser.id}`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Refresh-Token': session?.user?.refreshToken || '',
        },
      });

      const data = await response.json();
      
      if (data.success) {
        alert('User deleted successfully');
        setDeleteDialogOpen(false);
        setSelectedUser(null);
        fetchUsers(); // Refresh the list
      } else {
        alert(data.message || 'Failed to delete user');
      }
    } catch (error) {
      console.error('Error deleting user:', error);
      alert('Failed to delete user');
    }
  };

  // Get user type color
  const getUserTypeColor = (userType?: string) => {
    switch (userType) {
      case 'FIGHTER': return 'error';
      case 'COACH': return 'success';
      case 'PROFESSIONAL': return 'info';
      case 'GUEST_GENERAL_VISITOR': return 'default';
      default: return 'default';
    }
  };

  // Format user type label for display
  const getUserTypeLabel = (userType?: string) => {
    if (!userType) return 'Not set'
    if (userType === 'GUEST_GENERAL_VISITOR') return 'GUEST'
    return userType
  }

  useEffect(() => {
    if (session?.user?.sessionToken) {
      fetchUsers();
    } else if (session && !session.user?.sessionToken) {
      console.error('No session token found in session:', session);
      setLoading(false);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [page, search, userTypeFilter, session]);

  return (
    <Card>
      <CardContent>
        <Typography variant="h4" gutterBottom>
          User Management
        </Typography>

        {/* Filters */}
        <Box sx={{ display: 'flex', gap: 2, mb: 3, mt: 2 }}>
          <TextField
            placeholder="Search by name or email..."
            value={search}
            onChange={(e) => {
              setSearch(e.target.value);
              setPage(1); // Reset to first page on search
            }}
            variant="outlined"
            size="small"
            sx={{ flexGrow: 1 }}
            InputProps={{
              startAdornment: (
                <InputAdornment position="start">
                  <SearchIcon />
                </InputAdornment>
              ),
            }}
          />

          <Select
            value={userTypeFilter}
            onChange={(e) => {
              setUserTypeFilter(e.target.value);
              setPage(1);
            }}
            size="small"
            sx={{ minWidth: 200 }}
          >
            <MenuItem value="ALL">All User Types</MenuItem>
            <MenuItem value="GUEST_GENERAL_VISITOR">Guest</MenuItem>
            <MenuItem value="FIGHTER">Fighter</MenuItem>
            <MenuItem value="COACH">Coach</MenuItem>
            <MenuItem value="PROFESSIONAL">Professional</MenuItem>
          </Select>
        </Box>

        {/* Users Table */}
        {loading ? (
          <Typography>Loading...</Typography>
        ) : (
          <>
            <TableContainer component={Paper} sx={{ mt: 3 }}>
              <Table sx={{ minWidth: 650 }}>
                <TableHead>
                  <TableRow>
                    <TableCell><strong>Name</strong></TableCell>
                    <TableCell><strong>Email</strong></TableCell>
                    <TableCell><strong>Phone</strong></TableCell>
                    <TableCell><strong>User Type</strong></TableCell>
                    <TableCell><strong>City</strong></TableCell>
                    <TableCell align="center"><strong>Actions</strong></TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {users.map((user) => (
                    <TableRow key={user.id} hover>
                      <TableCell>
                        {[user.firstName, user.lastName].filter(Boolean).join(' ') || 
                          <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
                        }
                      </TableCell>
                      <TableCell>{user.email}</TableCell>
                      <TableCell>
                        {user.phoneNumber || 
                          <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
                        }
                      </TableCell>
                      <TableCell>
                        <Chip
                          label={getUserTypeLabel(user.userType)}
                          color={getUserTypeColor(user.userType)}
                          size="small"
                        />
                      </TableCell>
                      <TableCell>
                        {user.city || 
                          <span style={{ color: '#999', fontStyle: 'italic' }}>Not set</span>
                        }
                      </TableCell>
                      {/* Role column removed from table as requested */}
                      <TableCell align="center">
                        <IconButton
                          size="small"
                          color="primary"
                          onClick={() => fetchUserDetails(user.id)}
                        >
                          <ViewIcon />
                        </IconButton>
                        
                        {user.role !== 'ADMIN' && (
                          <IconButton
                            size="small"
                            color="error"
                            onClick={() => {
                              setSelectedUser(user);
                              setDeleteDialogOpen(true);
                            }}
                          >
                            <DeleteIcon />
                          </IconButton>
                        )}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </TableContainer>

            {/* Pagination */}
            <Box sx={{ display: 'flex', justifyContent: 'center', mt: 3 }}>
              <Pagination
                count={totalPages}
                page={page}
                onChange={(_, value) => setPage(value)}
                color="primary"
              />
            </Box>
          </>
        )}

        {/* Delete Confirmation Dialog */}
        <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
          <DialogTitle>Delete User</DialogTitle>
          <DialogContent>
            Are you sure you want to delete user <strong>{selectedUser?.email}</strong>?
            This action cannot be undone.
          </DialogContent>
          <DialogActions>
            <Button onClick={() => setDeleteDialogOpen(false)}>Cancel</Button>
            <Button onClick={handleDeleteUser} color="error" variant="contained">
              Delete
            </Button>
          </DialogActions>
        </Dialog>

        {/* View User Details Dialog */}
        <UserDetailsDialog
          open={viewDialogOpen}
          onClose={() => {
            setViewDialogOpen(false);
            setSelectedUserDetails(null);
          }}
          user={selectedUserDetails}
        />
      </CardContent>
    </Card>
  );
}

