'use client';

import { useState, useEffect } from 'react';

interface User {
  id: string;
  firstName?: string;
  lastName?: string;
  email: string;
  phoneNumber?: string;
  userType?: 'GUEST_GENERAL_VISITOR' | 'FIGHTER' | 'COACH' | 'PROFESSIONAL';
  role: 'ADMIN' | 'DEFAULT';
  city?: string;
  country?: string;
  createdAt: string;
}

export default function UserManagementSimple() {
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [search, setSearch] = useState('');
  const [userTypeFilter, setUserTypeFilter] = useState('ALL');
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<User | null>(null);
  const [viewDialogOpen, setViewDialogOpen] = useState(false);
  const [selectedUserDetails, setSelectedUserDetails] = useState<any>(null);

  const API_BASE_URL = process.env.NEXT_PUBLIC_API_URI || 'http://localhost:3024';

  // Fetch users
  const fetchUsers = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('sessionToken');
      
      const response = await fetch(
        `${API_BASE_URL}/user?page=${page}&limit=10&search=${search}&userType=${userTypeFilter}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        }
      );

      const data = await response.json();
      
      if (data.success) {
          // Hide ADMIN role users from the displayed list
          const visibleUsers = Array.isArray(data.data) ? data.data.filter((u: any) => u.role !== 'ADMIN') : []
          setUsers(visibleUsers);
        setTotalPages(data.pagination.totalPages);
      }
    } catch (error) {
      console.error('Error fetching users:', error);
      alert('Failed to fetch users');
    } finally {
      setLoading(false);
    }
  };

  // Fetch user details
  const fetchUserDetails = async (userId: string) => {
    try {
      const token = localStorage.getItem('sessionToken');
      
      const response = await fetch(`${API_BASE_URL}/user/${userId}`, {
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });

      const data = await response.json();
      
      if (data.success) {
        setSelectedUserDetails(data.data);
        setViewDialogOpen(true);
      }
    } catch (error) {
      console.error('Error fetching user details:', error);
      alert('Failed to fetch user details');
    }
  };

  // Delete user
  const handleDeleteUser = async () => {
    if (!selectedUser) return;

    try {
      const token = localStorage.getItem('sessionToken');
      
      const response = await fetch(`${API_BASE_URL}/user/${selectedUser.id}`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });

      const data = await response.json();
      
      if (data.success) {
        alert('User deleted successfully');
        setDeleteDialogOpen(false);
        setSelectedUser(null);
        fetchUsers();
      } else {
        alert(data.message || 'Failed to delete user');
      }
    } catch (error) {
      console.error('Error deleting user:', error);
      alert('Failed to delete user');
    }
  };

  // Get user type badge class
  const getUserTypeBadge = (userType?: string) => {
    const baseClass = 'px-2 py-1 rounded text-xs font-semibold';
    switch (userType) {
      case 'FIGHTER': return `${baseClass} bg-red-100 text-red-800`;
      case 'COACH': return `${baseClass} bg-green-100 text-green-800`;
      case 'PROFESSIONAL': return `${baseClass} bg-blue-100 text-blue-800`;
      case 'GUEST_GENERAL_VISITOR': return `${baseClass} bg-gray-100 text-gray-800`;
      default: return `${baseClass} bg-gray-100 text-gray-800`;
    }
  };

  const getUserTypeLabel = (userType?: string) => {
    if (!userType) return 'N/A'
    if (userType === 'GUEST_GENERAL_VISITOR') return 'Guest'
    return userType
  }

  useEffect(() => {
    fetchUsers();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [page, search, userTypeFilter]);

  return (
    <div className="container mx-auto p-4">
      <div className="bg-white rounded-lg shadow-lg p-6">
        <h1 className="text-3xl font-bold mb-6">User Management</h1>

        {/* Filters */}
        <div className="flex gap-4 mb-6">
          <input
            type="text"
            placeholder="Search by name or email..."
            value={search}
            onChange={(e) => {
              setSearch(e.target.value);
              setPage(1);
            }}
            className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
          />

          <select
            value={userTypeFilter}
            onChange={(e) => {
              setUserTypeFilter(e.target.value);
              setPage(1);
            }}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="ALL">All User Types</option>
            <option value="GUEST_GENERAL_VISITOR">Guest</option>
            <option value="FIGHTER">Fighter</option>
            <option value="COACH">Coach</option>
            <option value="PROFESSIONAL">Professional</option>
          </select>
        </div>

        {/* Users Table */}
        {loading ? (
          <div className="text-center py-8">Loading...</div>
        ) : (
          <>
            <div className="overflow-x-auto">
              <table className="min-w-full bg-white border border-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Name</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Phone</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User Type</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">City</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {users.map((user) => (
                    <tr key={user.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        {[user.firstName, user.lastName].filter(Boolean).join(' ') || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">{user.email}</td>
                      <td className="px-6 py-4 whitespace-nowrap">{user.phoneNumber || 'N/A'}</td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={getUserTypeBadge(user.userType)}>
                          {getUserTypeLabel(user.userType)}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">{user.city || 'N/A'}</td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <button
                          onClick={() => fetchUserDetails(user.id)}
                          className="text-blue-600 hover:text-blue-900 mr-4"
                        >
                          View
                        </button>
                        {user.role !== 'ADMIN' && (
                          <button
                            onClick={() => {
                              setSelectedUser(user);
                              setDeleteDialogOpen(true);
                            }}
                            className="text-red-600 hover:text-red-900"
                          >
                            Delete
                          </button>
                        )}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>

            {/* Pagination */}
            <div className="flex justify-center mt-6 gap-2">
              <button
                onClick={() => setPage(Math.max(1, page - 1))}
                disabled={page === 1}
                className="px-4 py-2 bg-blue-500 text-white rounded disabled:bg-gray-300 disabled:cursor-not-allowed"
              >
                Previous
              </button>
              <span className="px-4 py-2">
                Page {page} of {totalPages}
              </span>
              <button
                onClick={() => setPage(Math.min(totalPages, page + 1))}
                disabled={page === totalPages}
                className="px-4 py-2 bg-blue-500 text-white rounded disabled:bg-gray-300 disabled:cursor-not-allowed"
              >
                Next
              </button>
            </div>
          </>
        )}

        {/* Delete Confirmation Modal */}
        {deleteDialogOpen && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4">
              <h2 className="text-xl font-bold mb-4">Delete User</h2>
              <p className="mb-6">
                Are you sure you want to delete user <strong>{selectedUser?.email}</strong>?
                This action cannot be undone.
              </p>
              <div className="flex justify-end gap-4">
                <button
                  onClick={() => setDeleteDialogOpen(false)}
                  className="px-4 py-2 border border-gray-300 rounded hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button
                  onClick={handleDeleteUser}
                  className="px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
                >
                  Delete
                </button>
              </div>
            </div>
          </div>
        )}

        {/* View User Details Modal */}
        {viewDialogOpen && selectedUserDetails && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 max-w-4xl w-full mx-4 max-h-[90vh] overflow-y-auto">
              <h2 className="text-xl font-bold mb-4">User Details</h2>
              
                  <div className="grid grid-cols-2 gap-4 mb-6">
                <div>
                  <p className="text-sm text-gray-500">Name</p>
                  <p className="font-medium">
                    {[selectedUserDetails.firstName, selectedUserDetails.middleName, selectedUserDetails.lastName]
                      .filter(Boolean).join(' ') || 'N/A'}
                  </p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Email</p>
                  <p className="font-medium">{selectedUserDetails.email}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Phone</p>
                  <p className="font-medium">{selectedUserDetails.phoneNumber || 'N/A'}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">User Type</p>
                  <p className="font-medium">{getUserTypeLabel(selectedUserDetails.userType)}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Gender</p>
                  <p className="font-medium">{selectedUserDetails.gender || 'N/A'}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">City</p>
                  <p className="font-medium">{selectedUserDetails.city || 'N/A'}</p>
                </div>
              </div>

              {/* Documents */}
              {(selectedUserDetails.signingForms || selectedUserDetails.identityDocument || 
                selectedUserDetails.medicalDocument || selectedUserDetails.athleteCard || 
                selectedUserDetails.coachCertificate || selectedUserDetails.judgesCertificate) && (
                <div className="mb-6">
                  <h3 className="font-bold mb-2">Documents</h3>
                  <div className="flex flex-col gap-2">
                    {selectedUserDetails.signingForms && (
                      <a href={selectedUserDetails.signingForms} target="_blank" rel="noopener noreferrer"
                        className="text-blue-600 hover:underline">
                        📄 Signing Forms
                      </a>
                    )}
                    {selectedUserDetails.identityDocument && (
                      <a href={selectedUserDetails.identityDocument} target="_blank" rel="noopener noreferrer"
                        className="text-blue-600 hover:underline">
                        📄 Identity Document
                      </a>
                    )}
                    {selectedUserDetails.medicalDocument && (
                      <a href={selectedUserDetails.medicalDocument} target="_blank" rel="noopener noreferrer"
                        className="text-blue-600 hover:underline">
                        📄 Medical Document
                      </a>
                    )}
                    {selectedUserDetails.athleteCard && (
                      <a href={selectedUserDetails.athleteCard} target="_blank" rel="noopener noreferrer"
                        className="text-blue-600 hover:underline">
                        📄 Athlete Card
                      </a>
                    )}
                    {selectedUserDetails.coachCertificate && (
                      <a href={selectedUserDetails.coachCertificate} target="_blank" rel="noopener noreferrer"
                        className="text-blue-600 hover:underline">
                        📄 Coach Certificate
                      </a>
                    )}
                    {selectedUserDetails.judgesCertificate && (
                      <a href={selectedUserDetails.judgesCertificate} target="_blank" rel="noopener noreferrer"
                        className="text-blue-600 hover:underline">
                        📄 Judges Certificate
                      </a>
                    )}
                  </div>
                </div>
              )}

              <div className="flex justify-end">
                <button
                  onClick={() => {
                    setViewDialogOpen(false);
                    setSelectedUserDetails(null);
                  }}
                  className="px-4 py-2 bg-gray-200 rounded hover:bg-gray-300"
                >
                  Close
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}

